<?php

namespace App\Http\Controllers;

use App\Models\Service;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;
use DB;
use App\Traits\DeleteModelTrait;

class ServiceController extends Controller
{
    use DeleteModelTrait;

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'photo' => 'nullable|string',
                'icon' => 'nullable|string|max:100',
                'link' => 'nullable|string|max:255',
                'display_order' => 'nullable|integer|min:0',
                'status' => 'required|boolean',
            ]);

            $service = new Service();
            $service->fill($validatedData);
            $service->save();

            DB::commit();

            return Redirect::route('admin.services')->with('success', [
                'message' => 'Service created successfully',
                'check' => false
            ]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            return Redirect::back()->with('error', [
                'message' => 'Error: ' . $e->getMessage(),
                'check' => $e->getMessage()
            ]);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Service $service)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $service = Service::findOrFail($id);
        return response()->json($service);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            DB::beginTransaction();

            $validatedData = $request->validate([
                'name' => 'required|string|max:255',
                'description' => 'nullable|string',
                'photo' => 'nullable|string',
                'icon' => 'nullable|string|max:100',
                'link' => 'nullable|string|max:255',
                'display_order' => 'nullable|integer|min:0',
                'status' => 'required|boolean',
            ]);

            $service = Service::findOrFail($id);
            $service->fill($validatedData);
            
            if ($service->isDirty()) {
                $service->save();
            }

            DB::commit();

            return Redirect::back()->with('success', [
                'message' => 'Service updated successfully',
                'check' => false
            ]);
        } catch (ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            return Redirect::back()->with('error', [
                'message' => 'Error: ' . $e->getMessage(),
                'check' => $e->getMessage()
            ]);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        return $this->deleteModel($id, new Service());
    }

    /**
     * Fetch all services for admin panel
     */
    public function fetchServices()
    {
        $services = Service::orderBy('display_order')
            ->orderBy('created_at', 'desc')
            ->get();

        return Inertia::render('services/Service', [
            'services' => $services->map(function ($service) {
                return [
                    'id' => $service->id,
                    'name' => $service->name,
                    'description' => $service->description,
                    'photo' => $service->photo,
                    'icon' => $service->icon,
                    'link' => $service->link,
                    'display_order' => $service->display_order,
                    'status' => $service->status,
                    'created_at' => $service->created_at,
                ];
            }),
        ]);
    }

    /**
     * Upload service image
     */
    public function uploadImage(Request $request)
    {
        $request->validate([
            'file' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:5048',
        ]);

        $path = $request->file('file')->store('service-images', 'public');
        $filename = basename($path);

        return response()->json(['filename' => 'service-images/' . $filename]);
    }
}
