import React, { useState, useEffect } from "react";
import axios from "axios";
import { toast } from "react-hot-toast";

const MultiImageUploadInput = ({
  label = "Upload Images",
  inputId = "multiImageUpload",
  buttonText = "Choose images",
  uploadUrl,
  onSuccess,
  onError,
  presetPreviews = [],
}) => {
  const [files, setFiles] = useState([]);
  const [previews, setPreviews] = useState([]);
  const [isInitialLoad, setIsInitialLoad] = useState(true);

  // Load preset previews when component mounts or when they change
  useEffect(() => {
    // Always update previews when presetPreviews changes, even if empty
    setPreviews(presetPreviews);
    if (presetPreviews.length > 0) {
      setIsInitialLoad(false);
    } else {
      setIsInitialLoad(true);
    }
  }, [presetPreviews]);

  const handleFileChange = (e) => {
    const selectedFiles = Array.from(e.target.files);

    // Prevent selecting more than 8 images in total
    if (previews.length + selectedFiles.length > 8) {
      toast.error("You can only upload a maximum of 8 images.");
      e.target.value = "";
      return;
    }

    // Convert each file to a preview URL and store in state
    selectedFiles.forEach((file) => {
      const reader = new FileReader();
      reader.onload = () => {
        setPreviews((prev) => [...prev, reader.result]);
        setFiles((prev) => [...prev, file]);
      };
      reader.readAsDataURL(file);
    });

    // Clear the native file input so we can re-select the same file if needed
    e.target.value = "";
  };

  const handleRemoveImage = (index) => {
    // Calculate updated previews first (before state update)
    const updatedPreviews = previews.filter((_, i) => i !== index);
    
    // Remove the file and preview at the specified index
    setPreviews(updatedPreviews);
    
    // Only remove from files array if it's a newly added file
    if (index < files.length) {
      setFiles((prev) => prev.filter((_, i) => i !== index));
    }
    
    // If we've removed a preset preview, we should notify parent component
    // that the gallery has been modified
    if (isInitialLoad === false && presetPreviews.length > 0) {
      // Create filenames array from updated previews, stripping storage/ prefix
      const updatedFilenames = updatedPreviews
        .filter(url => url && url.includes('/storage/'))
        .map(url => url.replace('/storage/', ''));
      
      // Notify parent component with new filenames
      if (onSuccess) {
        onSuccess({ filenames: updatedFilenames });
      }
    }
  };

  const handleUpload = async () => {
    if (files.length === 0) {
      toast.error("No new images selected.");
      return;
    }

    const formData = new FormData();
    files.forEach((file) => {
      formData.append("files[]", file);
    });

    toast.promise(
      axios.post(uploadUrl, formData, {
        headers: { "Content-Type": "multipart/form-data" },
      }),
      {
        loading: "Uploading images...",
        success: (response) => {
          // After upload, combine existing presetPreviews with new uploaded files
          if (onSuccess) {
            // Get existing filenames from presetPreviews (removing /storage/ prefix)
            const existingFilenames = presetPreviews
              .filter(url => url.includes('/storage/'))
              .map(url => url.replace('/storage/', ''));
              
            // Combine existing filenames with newly uploaded ones
            const combinedFilenames = [...existingFilenames, ...response.data.filenames];
            
            // Call onSuccess with the combined filenames in the expected format
            onSuccess({ filenames: combinedFilenames });
          }
          
          // Clear the files array since they've been uploaded
          setFiles([]);
          
          return "Upload complete!";
        },
        error: (error) => {
          if (onError) onError(error);
          return "Upload failed!";
        },
      }
    );
  };

  return (
    <div className="mb-3">
      <div className="input-group mb-2">
        <div className="custom-file">
          <input
            type="file"
            id={inputId}
            multiple
            accept="image/*"
            className="custom-file-input"
            onChange={handleFileChange}
          />
          <label className="custom-file-label" htmlFor={inputId}>
            {files.length > 0
              ? `Choose Album images(600x600): ${files.length} new image(s) selected`
              : buttonText}
          </label>
        </div>
      </div>

      {/* Thumbnails of selected and existing images + remove button */}
      {previews.length > 0 && (
        <div className="d-flex flex-wrap gap-2 mb-2">
          {previews.map((preview, index) => (
            <div
              key={index}
              style={{ position: "relative", display: "inline-block" }}
            >
              <img
                src={preview}
                alt={`Preview ${index}`}
                className="img-thumbnail"
                style={{ width: "150px", height: "150px", objectFit: "cover" }}
              />
              <button
                type="button"
                className="btn btn-danger btn-sm"
                style={{
                  position: "absolute",
                  top: "5px",
                  right: "5px",
                  borderRadius: "50%",
                  padding: "0 6px",
                }}
                onClick={() => handleRemoveImage(index)}
              >
                &times;
              </button>
            </div>
          ))}
        </div>
      )}

      {/* Only show upload button if there are new files to upload */}
      {files.length > 0 && (
        <button type="button" className="btn btn-primary" onClick={handleUpload}>
          Upload Selected Images
        </button>
      )}
    </div>
  );
};

export default MultiImageUploadInput;