import Modal from "@admin/ui/Modal";
import { useForm } from "@inertiajs/react";
import { useEffect } from "react";
import { useFormResponse } from "@/hooks/useFormResponse";
import { formatDate, formatDateTime } from "@/utils/dateFormatter";

export default function ConfirmBooking({ isOpen, onClose, initialData }) {
  const { data, setData, put, processing,transform } = useForm({
    status: initialData?.status,
  });

  // Make sure to update the form state when initialData changes.
  useEffect(() => {
    if (initialData?.status) {
      setData("status", initialData.status);
    }
  }, [initialData, setData]);

  const handleConfirm = (event) => {
    event.preventDefault();

    // Toggle based on the initial status.
    const newStatus = initialData.status === "pending" ? "confirmed" : "pending";

    // Use transform to override the status just before submission.
    transform((data) => ({
      ...data,
      status: newStatus,
    }));

    const formResponse = useFormResponse(() => {
      onClose?.();
    });

    put(`/admin/confirm_booking/${initialData.id}`, {
      preserveScroll: true,
      onSuccess: formResponse.onSuccess,
      onError: formResponse.onError,
    });
  };

  return (
    <Modal
      isOpen={isOpen}
      onClose={onClose}
      title="Confirm Booking"
      size="large"
      primaryButtonText={
        initialData?.status === "confirmed" ? "Cancel Booking" : "Confirm Booking"
      }
      onPrimaryAction={handleConfirm}
      processing={processing}
    >
      {initialData && (
        <div className="alert alert-light p-4 shadow-sm" role="alert">
          <h4 className="text-dark">Booking Summary - #{initialData.booking_reference}</h4>
          <hr />
          <p>Full Name: {initialData.full_name}</p>
          <p>Email: {initialData.email || "N/A"}</p>
          {initialData.phone && <p>Phone: {initialData.phone}</p>}
          <p>Accommodation: {initialData.accommodation || 'N/A'}</p>
          {initialData.room && (
            <p>Room: <strong>{initialData.room}</strong></p>
          )}
          {initialData.check_in && initialData.check_out && (
            <>
              <p>Check-in: {formatDate(initialData.check_in)}</p>
              <p>Check-out: {formatDate(initialData.check_out)}</p>
            </>
          )}
          {initialData.event_start && initialData.event_end && (
            <>
              <p>Event Start: {formatDateTime(initialData.event_start)}</p>
              <p>Event End: {formatDateTime(initialData.event_end)}</p>
            </>
          )}
          <p>Guests: 
            {initialData.number_of_adults !== undefined && (
              <> {initialData.number_of_adults} Adult{initialData.number_of_adults !== 1 ? 's' : ''}</>
            )}
            {initialData.number_of_children !== undefined && initialData.number_of_children > 0 && (
              <>, {initialData.number_of_children} Child{initialData.number_of_children !== 1 ? 'ren' : ''}</>
            )}
            {(!initialData.number_of_adults && !initialData.number_of_children) && (
              <> {initialData.number_of_people || 'N/A'}</>
            )}
          </p>
          <p>Price: ${initialData.price}</p>

          {initialData.special_request && (
            <p>Special Request: {initialData.special_request}</p>
          )}
          {/* <p className="text-dark">Destination: {initialData.destination || "N/A"}</p> */}
          <p>
            Status:{initialData.status}
          </p>
        </div>
      )}
    </Modal>
  );
}
