import Card from "@admin/ui/Card";
import Col from "@admin/ui/Col";
import FormGroup from "@admin/ui/forms/FormGroup";
import Switch from "@admin/ui/forms/Switch";
import FileUploadInput from "@admin/ui/forms/FileUploadInput";
import MultiImageUploadInput from "@admin/ui/forms/MultiImageUploadInput";
import Row from "@admin/ui/Row";
import { useForm } from "@inertiajs/react";
import { useFormResponse } from '@/hooks/useFormResponse';
import { Toaster } from "react-hot-toast";
import { useState } from "react";

export default function EventRoom(){
    const {data, setData, post, processing, errors, reset} = useForm({
        'name': '',
        'primary_photo': '',
        'price_fbu': '',
        'price_usd': '',
        'price_eur': '',
        'short_description': '',
        'full_description': '',
        'capacity': '',
        'layouts': [],
        'equipment': [],
        'gallery': [],
        'status': true,
    });

    // Store raw text for layouts and equipment to allow typing commas
    const [layoutsText, setLayoutsText] = useState('');
    const [equipmentText, setEquipmentText] = useState('');

    const handlePrimaryPhotoSuccess = (filename) => {
        console.log('Primary photo uploaded:', filename);
        setData('primary_photo', filename);
    };

    const handleGallerySuccess = (response) => {
        // Handle the response - it should have a filenames property
        const filenames = response?.filenames || (Array.isArray(response) ? response : []);
        setData('gallery', filenames);
        console.log('Gallery updated:', filenames); // Debug log
    };

    const handleError = (error) => {
        console.error("Upload error:", error);
    };

    const handleLayoutsChange = (e) => {
        const value = e.target.value;
        // Store raw text to allow typing commas
        setLayoutsText(value);
        // Convert to array for data storage
        const layoutsArray = value.split(',')
            .map(item => item.trim())
            .filter(item => item.length > 0);
        setData('layouts', layoutsArray);
    };

    const handleLayoutsBlur = () => {
        // On blur, ensure the text matches the array
        setLayoutsText(data.layouts.join(', '));
    };

    const handleEquipmentChange = (e) => {
        const value = e.target.value;
        // Store raw text to allow typing commas
        setEquipmentText(value);
        // Convert to array for data storage
        const equipmentArray = value.split(',')
            .map(item => item.trim())
            .filter(item => item.length > 0);
        setData('equipment', equipmentArray);
    };

    const handleEquipmentBlur = () => {
        // On blur, ensure the text matches the array
        setEquipmentText(data.equipment.join(', '));
    };

    const handleSubmit = (event) => {
        event.preventDefault();
        console.log('Submitting event room with gallery:', data.gallery);
        console.log('Primary photo:', data.primary_photo);
        console.log('Full form data:', data);
        
        const formResponse = useFormResponse(reset);
        post('/admin/event_rooms/store', {
            ...formResponse,
            preserveScroll: true,
        });
    }

    return(
        <>
        <Toaster />
        <Row md="12">
            <Col lg="12">
            <h2 className="page-title">Create New Event Room</h2>
            <p className="text-muted">Add conference and event room details</p>
            <Row>
            <Col lg="12">
            <Card md="12" title="Event Room Details">
            <form onSubmit={handleSubmit}>
                <Row>
                    <Col md="6">
                        <FormGroup
                            id="name"
                            label="Room Name"
                            type="text"
                            value={data.name}
                            onChange={(e) => setData("name", e.target.value)}
                            required
                            invalidFeedback={errors.name}
                        />

                        <FormGroup 
                            label="Price (FBU)" 
                            type="number"
                            value={data.price_fbu}
                            onChange={(e) => setData('price_fbu', e.target.value)}
                            helpText="Enter price in FBU"
                            invalidFeedback={errors.price_fbu}
                            min="0"
                            step="0.01"
                        />

                        <FormGroup 
                            label="Price (USD)" 
                            type="number"
                            value={data.price_usd}
                            onChange={(e) => setData('price_usd', e.target.value)}
                            helpText="Enter price in USD"
                            invalidFeedback={errors.price_usd}
                            min="0"
                            step="0.01"
                        />

                        <FormGroup 
                            label="Price (EUR)" 
                            type="number"
                            value={data.price_eur}
                            onChange={(e) => setData('price_eur', e.target.value)}
                            helpText="Enter price in EUR"
                            invalidFeedback={errors.price_eur}
                            min="0"
                            step="0.01"
                        />

                        <FormGroup
                            id="capacity"
                            label="Capacity"
                            type="number"
                            value={data.capacity}
                            onChange={(e) => setData("capacity", e.target.value)}
                            helpText="Maximum number of people"
                            invalidFeedback={errors.capacity}
                            min="1"
                        />

                        <FormGroup
                            id="layouts"
                            label="Layouts"
                            as="textarea"
                            value={layoutsText}
                            helpText="Enter layouts separated by commas (e.g., U-shape, Theatre, Boardroom)"
                            onChange={handleLayoutsChange}
                            onBlur={handleLayoutsBlur}
                            rows="3"
                            invalidFeedback={errors.layouts}
                        />
                        {data.layouts.length > 0 && (
                            <div className="mb-2">
                                <small className="text-muted">Current layouts: </small>
                                {data.layouts.map((layout, index) => (
                                    <span key={index} className="badge badge-secondary mr-1">{layout}</span>
                                ))}
                            </div>
                        )}

                        <FormGroup
                            id="equipment"
                            label="Equipment"
                            as="textarea"
                            value={equipmentText}
                            helpText="Enter equipment separated by commas (e.g., Projector, Mic, Speakers, Whiteboard)"
                            onChange={handleEquipmentChange}
                            onBlur={handleEquipmentBlur}
                            rows="3"
                            invalidFeedback={errors.equipment}
                        />
                        {data.equipment.length > 0 && (
                            <div className="mb-2">
                                <small className="text-muted">Current equipment: </small>
                                {data.equipment.map((item, index) => (
                                    <span key={index} className="badge badge-secondary mr-1">{item}</span>
                                ))}
                            </div>
                        )}
                    </Col>
                    <Col md="6">
                        <FormGroup
                            id="short_description"
                            label="Short Description"
                            as="textarea"
                            value={data.short_description}
                            helpText="Provide a brief description (minimum 20 characters)"
                            onChange={(e) => setData("short_description", e.target.value)}
                            rows="4"
                            invalidFeedback={errors.short_description}
                        />

                        <FormGroup
                            id="full_description"
                            label="Full Description"
                            as="textarea"
                            value={data.full_description}
                            helpText="Detailed description for the event room"
                            onChange={(e) => setData("full_description", e.target.value)}
                            rows="5"
                            invalidFeedback={errors.full_description}
                        />

                        <label className="label">Gallery Photos (1123x700)px</label>
                        <MultiImageUploadInput
                            label="Upload Gallery Images (1123x700)px"
                            inputId="galleryUpload"
                            buttonText="Choose gallery images (1123x700)"
                            uploadUrl="/admin/api/event_room/upload_multiple"
                            onSuccess={handleGallerySuccess}
                            onError={handleError}
                            presetPreviews={data.gallery.map(img => `/storage/${img}`)}
                        />
                        <small className="text-muted d-block mt-1">Recommended size: 1123x700 pixels</small>
                        {errors.gallery && (
                            <div className="invalid-feedback d-block">{errors.gallery}</div>
                        )}
                    </Col>
                </Row>

                <Row>
                    <Col md="6">
                        <FileUploadInput 
                            label="Upload Primary Photo (800x571)px" 
                            buttonText={"Choose file (800x571)"}
                            uploadUrl="/admin/api/event_room/upload_image"
                            onSuccess={handlePrimaryPhotoSuccess}
                            onError={handleError}
                            invalidFeedback={errors.primary_photo}
                        />
                        {data.primary_photo && (
                            <img 
                                src={`/storage/${data.primary_photo}`} 
                                alt="Primary photo preview" 
                                className="img-thumbnail mt-2"
                                style={{ maxWidth: '300px', maxHeight: '200px' }}
                            />
                        )}
                    </Col>
                </Row>

                <Switch
                    id="statusSwitch"
                    label="Active"
                    checked={data.status}
                    onChange={({ target: { checked } }) =>
                        setData(prevData => ({ ...prevData, status: checked }))
                    }                  
                />
                <br/>
                <button type="submit" className="btn btn-primary" disabled={processing}>
                    Save Event Room
                </button>
            </form>
            </Card>
            </Col>
            </Row>
            </Col>
        </Row>
        </>
    );
}

