import FormGroup from "@admin/ui/forms/FormGroup";
import Switch from "@admin/ui/forms/Switch";
import Modal from "@admin/ui/Modal";
import Row from "@admin/ui/Row";
import Col from "@admin/ui/Col";
import { useEffect, useState } from "react";
import { useForm, router } from "@inertiajs/react";
import { useFormResponse } from '@/hooks/useFormResponse';

export default function CreateEditHomeVideo({ isOpen, onClose, initialData }) {
    const [uploadProgress, setUploadProgress] = useState(null);
    const { data, setData, processing, errors, reset } = useForm({
        title: '',
        description: '',
        video_url: '',
        display_order: '',
        status: 1,
    });

    useEffect(() => {
        if (initialData) {
            setData({
                title: initialData?.title || '',
                description: initialData?.description || '',
                video_url: initialData?.video_url || '',
                display_order: initialData?.display_order || '',
                status: initialData?.status === 1 || initialData?.status === true ? 1 : 0,
            });
        } else {
            setData({
                title: '',
                description: '',
                video_url: '',
                display_order: '',
                status: 1,
            });
        }
    }, [initialData]);

    const handleSave = async (event) => {
        if (event?.preventDefault) event.preventDefault();
        
        const formResponse = useFormResponse(() => {
            onClose?.();
        });

        // Build FormData manually for proper handling
        const formData = new FormData();
        formData.append('title', data.title || '');
        formData.append('description', data.description || '');
        formData.append('video_url', data.video_url || '');
        formData.append('display_order', data.display_order || '');
        formData.append('status', data.status === 1 || data.status === true ? '1' : '0');

        const url = initialData 
            ? `/admin/api/update_homepage_video/${initialData.id}`
            : '/admin/api/create_homepage_video';

        router.post(url, formData, {
            preserveScroll: true,
            forceFormData: true,
            onSuccess: formResponse.onSuccess,
            onError: formResponse.onError,
            onProgress: (event) => {
                setUploadProgress(event?.percentage ?? null);
            },
        });
    };

    return (
        <Modal
            isOpen={isOpen}
            onClose={onClose}
            title={initialData ? `Edit ${initialData.title}` : 'Add Homepage Video'}
            size="large"
            primaryButtonText="Save Changes"
            onPrimaryAction={handleSave}
            processing={processing}
        >
            <form onSubmit={handleSave}>
                <Row>
                    <Col md="6">
                        <FormGroup
                            id="title"
                            label="Title"
                            type="text"
                            value={data.title}
                            onChange={(e) => setData('title', e.target.value)}
                            required
                            invalidFeedback={errors.title}
                        />

                        <FormGroup
                            id="display_order"
                            label="Display Order"
                            type="number"
                            value={data.display_order}
                            onChange={(e) => setData('display_order', e.target.value)}
                            helpText="Optional: lower numbers appear first"
                            invalidFeedback={errors.display_order}
                            min="0"
                        />

                        <FormGroup
                            id="video_url"
                            label="Video URL"
                            type="url"
                            value={data.video_url}
                            onChange={(e) => setData('video_url', e.target.value)}
                            helpText="Paste a YouTube/Vimeo/direct video link."
                            required
                            invalidFeedback={errors.video_url}
                        />
                    </Col>
                    <Col md="6">
                        <FormGroup
                            id="description"
                            label="Description"
                            as="textarea"
                            rows="8"
                            value={data.description}
                            onChange={(e) => setData('description', e.target.value)}
                            helpText="Short description shown with the video."
                            invalidFeedback={errors.description}
                        />

                        <Switch
                            id="statusSwitch"
                            label="Active"
                            checked={data.status === 1 || data.status === true}
                            onChange={({ target: { checked } }) =>
                                setData('status', checked ? 1 : 0)
                            }
                        />
                        {errors.status && <div className="invalid-feedback d-block">{errors.status}</div>}
                    </Col>
                </Row>
            </form>
        </Modal>
    );
}
