import Col from "@admin/ui/Col";
import Row from "@admin/ui/Row";
import StyledTable from "@admin/ui/StyledTable";
import { useState } from "react";
import DeleteConfirmationModal from "@admin/ui/DeleteConfirmationModal";
import { toast, Toaster } from "react-hot-toast";
import CreateEditHomeVideo from "./CreateEditHomeVideo";

export default function HomeVideos({ videos = [] }) {
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [initialData, setInitialData] = useState(null);
    const [deleteModal, setDeleteModal] = useState({ show: false, video: null });

    const columns = [
        { header: '', key: 'checkbox' },
        { header: 'Title', key: 'title' },
        { header: 'Source', key: 'source_type' },
        { header: 'Display Order', key: 'display_order' },
        { header: 'Status', key: 'status' },
    ];

    const handleCreate = () => {
        setInitialData(null);
        setIsModalOpen(true);
    };

    const handleEdit = (row) => {
        fetch(`/admin/edit_homepage_video/${row.id}`)
            .then((response) => {
                if (!response.ok) throw new Error('Network response was not ok');
                return response.json();
            })
            .then((data) => {
                setInitialData(data);
                setIsModalOpen(true);
            })
            .catch((error) => {
                console.error('Error fetching video:', error);
                toast.error('Failed to load video details');
            });
    };

    const handleDelete = (row) => {
        setDeleteModal({ show: true, video: row });
    };

    const handleCloseModal = () => {
        setIsModalOpen(false);
        setInitialData(null);
    };

    const enhanceRows = (items) => (items || []).map((item) => ({
        ...item,
        status: !!item.status,
        source_type: item.video_url ? 'Link' : (item.video_path ? 'Upload' : 'None'),
    }));

    return (
        <>
            <Toaster />
            <Row md="12">
                <Col lg="12">
                    <h2 className="page-title">Homepage Videos</h2>
                    <p className="text-muted">Upload or link videos for the homepage (up to two recommended).</p>
                    <button className="btn mb-4 btn-primary" onClick={handleCreate}>
                        Add Homepage Video
                    </button>

                    <Row>
                        <Col lg="12">
                            <StyledTable
                                data={enhanceRows(videos)}
                                columns={columns}
                                itemsPerPage={10}
                                title="Videos"
                                onEdit={handleEdit}
                                onDelete={handleDelete}
                                noDataText="No videos found"
                            />
                        </Col>
                    </Row>
                </Col>
            </Row>

            <DeleteConfirmationModal
                isOpen={deleteModal.show}
                onClose={() => setDeleteModal({ show: false, video: null })}
                onConfirmDelete={() => {
                    toast.success("Video Deleted Successfully");
                }}
                itemName={deleteModal.video?.title || 'this video'}
                itemType="homepage video"
                deleteEndpoint={`/admin/api/delete_homepage_video/${deleteModal?.video?.id}`}
            />

            <CreateEditHomeVideo
                isOpen={isModalOpen}
                onClose={handleCloseModal}
                initialData={initialData}
            />
        </>
    );
}
