import FormGroup from "@admin/ui/forms/FormGroup";
import Modal from "@admin/ui/Modal";
import Row from "@admin/ui/Row";
import Col from "@admin/ui/Col";
import { useEffect } from "react";
import { useFormResponse } from '@/hooks/useFormResponse';
import { useForm } from "@inertiajs/react";

export default function EditPricingRule({ isOpen, onClose, initialData }) {
    const { data, setData, put, processing, errors } = useForm({
        accommodation_id: '',
        base_price_bif: '',
        base_price_usd: '',
        base_price_eur: '',
        additional_adult_bif: '',
        additional_adult_usd: '',
        additional_adult_eur: '',
        child_surcharge_bif: '',
        child_surcharge_usd: '',
        child_surcharge_eur: '',
    });

    useEffect(() => {
        if (initialData) {
            setData({
                accommodation_id: initialData?.accommodation_id || '',
                base_price_bif: initialData?.base_price_bif || '',
                base_price_usd: initialData?.base_price_usd || '',
                base_price_eur: initialData?.base_price_eur || '',
                additional_adult_bif: initialData?.additional_adult_bif || 0,
                additional_adult_usd: initialData?.additional_adult_usd || 0,
                additional_adult_eur: initialData?.additional_adult_eur || 0,
                child_surcharge_bif: initialData?.child_surcharge_bif || 0,
                child_surcharge_usd: initialData?.child_surcharge_usd || 0,
                child_surcharge_eur: initialData?.child_surcharge_eur || 0,
            });
        }
    }, [initialData, setData]);

    const handleSave = (event) => {
        event.preventDefault();
        const formResponse = useFormResponse(() => {
            onClose?.();
        });
        put(`/admin/api/update_pricing_rule/${initialData?.accommodation_id}`, {
            preserveScroll: true,
            onSuccess: formResponse.onSuccess,
            onError: formResponse.onError,
        });
    };

    if (!initialData) return null;

    return (
        <Modal
            isOpen={isOpen}
            onClose={onClose}
            title={`Edit Pricing Rules: ${initialData?.accommodation_name || ''}`}
            size="large"
            primaryButtonText="Save Changes"
            onPrimaryAction={handleSave}
            processing={processing}
        >
            <form onSubmit={handleSave}>
                <Row>
                    <Col md="12">
                        <div className="alert alert-info">
                            <strong>Base Price:</strong> Price for 1 adult (per night)
                        </div>
                    </Col>
                </Row>

                <Row>
                    <Col md="4">
                        <FormGroup
                            id="base_price_bif"
                            label="Base Price (BIF)"
                            type="number"
                            step="0.01"
                            value={data.base_price_bif}
                            onChange={(e) => setData("base_price_bif", e.target.value)}
                            required
                            invalidFeedback={errors.base_price_bif}
                            helpText="Base price for 1 adult in BIF"
                        />
                    </Col>
                    <Col md="4">
                        <FormGroup
                            id="base_price_usd"
                            label="Base Price (USD)"
                            type="number"
                            step="0.01"
                            value={data.base_price_usd}
                            onChange={(e) => setData("base_price_usd", e.target.value)}
                            required
                            invalidFeedback={errors.base_price_usd}
                            helpText="Base price for 1 adult in USD"
                        />
                    </Col>
                    <Col md="4">
                        <FormGroup
                            id="base_price_eur"
                            label="Base Price (EUR)"
                            type="number"
                            step="0.01"
                            value={data.base_price_eur}
                            onChange={(e) => setData("base_price_eur", e.target.value)}
                            required
                            invalidFeedback={errors.base_price_eur}
                            helpText="Base price for 1 adult in EUR"
                        />
                    </Col>
                </Row>

                <Row>
                    <Col md="12">
                        <div className="alert alert-info mt-3">
                            <strong>Additional Adult Charge:</strong> Extra charge per additional adult (beyond the first one)
                        </div>
                    </Col>
                </Row>

                <Row>
                    <Col md="4">
                        <FormGroup
                            id="additional_adult_bif"
                            label="Additional Adult (BIF)"
                            type="number"
                            step="0.01"
                            value={data.additional_adult_bif}
                            onChange={(e) => setData("additional_adult_bif", e.target.value)}
                            invalidFeedback={errors.additional_adult_bif}
                            helpText="Extra charge per additional adult in BIF (0 = no charge)"
                        />
                    </Col>
                    <Col md="4">
                        <FormGroup
                            id="additional_adult_usd"
                            label="Additional Adult (USD)"
                            type="number"
                            step="0.01"
                            value={data.additional_adult_usd}
                            onChange={(e) => setData("additional_adult_usd", e.target.value)}
                            invalidFeedback={errors.additional_adult_usd}
                            helpText="Extra charge per additional adult in USD"
                        />
                    </Col>
                    <Col md="4">
                        <FormGroup
                            id="additional_adult_eur"
                            label="Additional Adult (EUR)"
                            type="number"
                            step="0.01"
                            value={data.additional_adult_eur}
                            onChange={(e) => setData("additional_adult_eur", e.target.value)}
                            invalidFeedback={errors.additional_adult_eur}
                            helpText="Extra charge per additional adult in EUR"
                        />
                    </Col>
                </Row>

                <Row>
                    <Col md="12">
                        <div className="alert alert-info mt-3">
                            <strong>Child Surcharge:</strong> Extra charge per child
                        </div>
                    </Col>
                </Row>

                <Row>
                    <Col md="4">
                        <FormGroup
                            id="child_surcharge_bif"
                            label="Child Surcharge (BIF)"
                            type="number"
                            step="0.01"
                            value={data.child_surcharge_bif}
                            onChange={(e) => setData("child_surcharge_bif", e.target.value)}
                            invalidFeedback={errors.child_surcharge_bif}
                            helpText="Extra charge per child in BIF (0 = no charge)"
                        />
                    </Col>
                    <Col md="4">
                        <FormGroup
                            id="child_surcharge_usd"
                            label="Child Surcharge (USD)"
                            type="number"
                            step="0.01"
                            value={data.child_surcharge_usd}
                            onChange={(e) => setData("child_surcharge_usd", e.target.value)}
                            invalidFeedback={errors.child_surcharge_usd}
                            helpText="Extra charge per child in USD"
                        />
                    </Col>
                    <Col md="4">
                        <FormGroup
                            id="child_surcharge_eur"
                            label="Child Surcharge (EUR)"
                            type="number"
                            step="0.01"
                            value={data.child_surcharge_eur}
                            onChange={(e) => setData("child_surcharge_eur", e.target.value)}
                            invalidFeedback={errors.child_surcharge_eur}
                            helpText="Extra charge per child in EUR"
                        />
                    </Col>
                </Row>
            </form>
        </Modal>
    );
}



