import Card from "@admin/ui/Card";
import Col from "@admin/ui/Col";
import FormGroup from "@admin/ui/forms/FormGroup";
import Switch from "@admin/ui/forms/Switch";
import FileUploadInput from "@admin/ui/forms/FileUploadInput";
import Row from "@admin/ui/Row";
import { useForm } from "@inertiajs/react";
import { useFormResponse } from '@/hooks/useFormResponse';
import { Toaster } from "react-hot-toast";

export default function Service() {
    const { data, setData, post, processing, errors, reset } = useForm({
        name: '',
        description: '',
        photo: '',
        icon: '',
        link: '',
        display_order: 0,
        status: true,
    });

    const handleSuccess = (filename) => {
        setData('photo', filename);
    };

    const handleError = (error) => {
        console.error("Upload error:", error);
    };

    const handleSubmit = (event) => {
        event.preventDefault();
        const formResponse = useFormResponse(reset);
        post('/admin/services/store', {
            ...formResponse,
            preserveScroll: true,
        });
    };

    return (
        <>
            <Toaster />
            <Row md="12">
                <Col lg="12">
                    <h2 className="page-title">Create New Service</h2>
                    <p className="text-muted">Add a new service to be displayed on the frontend</p>
                    <Row>
                        <Col lg="12">
                            <Card md="8" title="Service Details">
                                <form onSubmit={handleSubmit}>
                                    <FormGroup
                                        label="Service Name"
                                        value={data.name}
                                        onChange={(e) => setData('name', e.target.value)}
                                        invalidFeedback={errors.name}
                                        required
                                        helpText="Enter the service name (e.g., Swimming Pool, Catering Services)"
                                    />

                                    <FormGroup
                                        id="description"
                                        label="Description"
                                        as="textarea"
                                        value={data.description}
                                        onChange={(e) => setData('description', e.target.value)}
                                        rows="4"
                                        invalidFeedback={errors.description}
                                        helpText="Describe the service in detail"
                                    />

                                    <FormGroup
                                        id="icon"
                                        label="Icon (Optional)"
                                        type="text"
                                        value={data.icon}
                                        onChange={(e) => setData('icon', e.target.value)}
                                        invalidFeedback={errors.icon}
                                        helpText="Icon class name (e.g., fe fe-activity, or use an icon library)"
                                    />

                                    <FormGroup
                                        id="link"
                                        label="Link (Optional)"
                                        type="text"
                                        value={data.link}
                                        onChange={(e) => setData('link', e.target.value)}
                                        invalidFeedback={errors.link}
                                        helpText="Link to related page (e.g., /rooms, /dining, /meetings-events)"
                                    />

                                    <FormGroup
                                        id="display_order"
                                        label="Display Order"
                                        type="number"
                                        value={data.display_order}
                                        onChange={(e) => setData('display_order', parseInt(e.target.value) || 0)}
                                        invalidFeedback={errors.display_order}
                                        helpText="Lower numbers appear first (0 = first)"
                                    />

                                    <br />
                                    <FileUploadInput
                                        label="Upload Service Photo (800×600px)"
                                        buttonText="Choose Photo"
                                        inputId="servicePhotoUpload"
                                        uploadUrl="/admin/api/service/upload_image"
                                        onSuccess={handleSuccess}
                                        onError={handleError}
                                        helpText="Recommended size: 800×600px (4:3 aspect ratio). Images will be displayed at 250px height with responsive width."
                                    />

                                    <Switch
                                        id="statusSwitch"
                                        label="Active"
                                        checked={data.status}
                                        onChange={({ target: { checked } }) =>
                                            setData(prevData => ({ ...prevData, status: checked }))
                                        }
                                    />
                                    <br />
                                    <button type="submit" className="btn btn-primary" disabled={processing}>
                                        {processing ? 'Creating...' : 'Create Service'}
                                    </button>
                                    <a href="/admin/services" className="btn btn-secondary ml-2">
                                        Cancel
                                    </a>
                                </form>
                            </Card>
                        </Col>
                    </Row>
                </Col>
            </Row>
        </>
    );
}

