import { Link } from '@inertiajs/react';

export default function Video({ videos = [] }) {
    // Extract YouTube video ID from URL
    const getYouTubeId = (url) => {
        if (!url) return null;
        const regExp = /^.*(youtu\.be\/|v\/|u\/\w\/|embed\/|watch\?v=|&v=)([^#\&\?]*).*/;
        const match = url.match(regExp);
        return match && match[2].length === 11 ? match[2] : null;
    };

    // Extract Vimeo video ID from URL
    const getVimeoId = (url) => {
        if (!url) return null;
        const regExp = /(?:www\.|player\.)?vimeo\.com\/(?:channels\/(?:\w+\/)?|groups\/([^\/]*)\/videos\/|album\/(\d+)\/video\/|video\/|)(\d+)(?:$|\/)/;
        const match = url.match(regExp);
        return match && match[3] ? match[3] : null;
    };

    // Detect if URL is YouTube
    const isYouTube = (url) => {
        return url && (url.includes('youtube.com') || url.includes('youtu.be'));
    };

    // Detect if URL is Vimeo
    const isVimeo = (url) => {
        return url && url.includes('vimeo.com');
    };

    // Get video items - use videos if available, otherwise return empty
    const getVideoItems = () => {
        if (videos && videos.length > 0) {
            return videos.map((video) => ({
                id: video.id,
                title: video.title,
                description: video.description,
                video_path: video.video_path,
                video_url: video.video_url,
                source: video.video_path ? 'upload' : 'url',
            }));
        }
        return [];
    };

    const videoItems = getVideoItems();

    // Don't render section if no videos
    if (videoItems.length === 0) {
        return null;
    }

    // Render video player based on source type
    const renderVideoPlayer = (video) => {
        const youtubeId = getYouTubeId(video.video_url);
        const vimeoId = getVimeoId(video.video_url);

        if (youtubeId) {
            return (
                <iframe
                    width="100%"
                    height="100%"
                    src={`https://www.youtube.com/embed/${youtubeId}`}
                    title={video.title}
                    frameBorder="0"
                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                    allowFullScreen
                    style={{
                        position: 'absolute',
                        top: 0,
                        left: 0,
                    }}
                />
            );
        }

        if (vimeoId) {
            return (
                <iframe
                    width="100%"
                    height="100%"
                    src={`https://player.vimeo.com/video/${vimeoId}`}
                    title={video.title}
                    frameBorder="0"
                    allow="autoplay; fullscreen; picture-in-picture"
                    allowFullScreen
                    style={{
                        position: 'absolute',
                        top: 0,
                        left: 0,
                    }}
                />
            );
        }

        // For direct video URLs or uploaded videos, use HTML5 video tag
        return (
            <video
                style={{
                    position: 'absolute',
                    top: 0,
                    left: 0,
                    width: '100%',
                    height: '100%',
                }}
                controls
                controlsList="nodownload"
                crossOrigin="anonymous"
            >
                <source
                    src={video.source === 'upload' ? video.video_path : video.video_url}
                    type={video.source === 'upload' ? 'video/mp4' : ''}
                />
                Your browser does not support the video tag.
            </video>
        );
    };

    return (
        <>
            <section className="section bg-light">
                <div className="container">
                    <div className="row justify-content-center text-center mb-5">
                        <div className="col-md-7">
                            <h2 className="heading" data-aos="fade-up">
                                Featured Videos
                            </h2>
                            <p data-aos="fade-up" data-aos-delay="100">
                                Explore our latest video content showcasing our services and experiences
                            </p>
                        </div>
                    </div>

                    <div className="row">
                        {videoItems.map((video, index) => (
                            <div
                                key={video.id}
                                className="col-md-6 mb-4"
                                data-aos="fade-up"
                                data-aos-delay={index * 100}
                            >
                                <div className="card h-100 shadow-sm border-0">
                                    {/* Video Player */}
                                    <div
                                        className="position-relative"
                                        style={{
                                            paddingBottom: '56.25%',
                                            height: 0,
                                            overflow: 'hidden',
                                        }}
                                    >
                                        {renderVideoPlayer(video)}
                                    </div>

                                    {/* Card Content */}
                                    <div className="card-body d-flex flex-column">
                                        {/* <h5 className="card-title">{video.title}</h5> */}
                                        <p className="card-text flex-grow-1">
                                            {video.description}
                                        </p>
                                    </div>
                                </div>
                            </div>
                        ))}
                    </div>
                </div>
            </section>
        </>
    );
}
